﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/redshift-data/RedshiftDataAPIServiceRequest.h>
#include <aws/redshift-data/RedshiftDataAPIService_EXPORTS.h>
#include <aws/redshift-data/model/ResultFormatString.h>
#include <aws/redshift-data/model/SqlParameter.h>

#include <utility>

namespace Aws {
namespace RedshiftDataAPIService {
namespace Model {

/**
 */
class ExecuteStatementRequest : public RedshiftDataAPIServiceRequest {
 public:
  AWS_REDSHIFTDATAAPISERVICE_API ExecuteStatementRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ExecuteStatement"; }

  AWS_REDSHIFTDATAAPISERVICE_API Aws::String SerializePayload() const override;

  AWS_REDSHIFTDATAAPISERVICE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The SQL statement text to run. </p>
   */
  inline const Aws::String& GetSql() const { return m_sql; }
  inline bool SqlHasBeenSet() const { return m_sqlHasBeenSet; }
  template <typename SqlT = Aws::String>
  void SetSql(SqlT&& value) {
    m_sqlHasBeenSet = true;
    m_sql = std::forward<SqlT>(value);
  }
  template <typename SqlT = Aws::String>
  ExecuteStatementRequest& WithSql(SqlT&& value) {
    SetSql(std::forward<SqlT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The cluster identifier. This parameter is required when connecting to a
   * cluster and authenticating using either Secrets Manager or temporary
   * credentials. </p>
   */
  inline const Aws::String& GetClusterIdentifier() const { return m_clusterIdentifier; }
  inline bool ClusterIdentifierHasBeenSet() const { return m_clusterIdentifierHasBeenSet; }
  template <typename ClusterIdentifierT = Aws::String>
  void SetClusterIdentifier(ClusterIdentifierT&& value) {
    m_clusterIdentifierHasBeenSet = true;
    m_clusterIdentifier = std::forward<ClusterIdentifierT>(value);
  }
  template <typename ClusterIdentifierT = Aws::String>
  ExecuteStatementRequest& WithClusterIdentifier(ClusterIdentifierT&& value) {
    SetClusterIdentifier(std::forward<ClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name or ARN of the secret that enables access to the database. This
   * parameter is required when authenticating using Secrets Manager. </p>
   */
  inline const Aws::String& GetSecretArn() const { return m_secretArn; }
  inline bool SecretArnHasBeenSet() const { return m_secretArnHasBeenSet; }
  template <typename SecretArnT = Aws::String>
  void SetSecretArn(SecretArnT&& value) {
    m_secretArnHasBeenSet = true;
    m_secretArn = std::forward<SecretArnT>(value);
  }
  template <typename SecretArnT = Aws::String>
  ExecuteStatementRequest& WithSecretArn(SecretArnT&& value) {
    SetSecretArn(std::forward<SecretArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The database user name. This parameter is required when connecting to a
   * cluster as a database user and authenticating using temporary credentials. </p>
   */
  inline const Aws::String& GetDbUser() const { return m_dbUser; }
  inline bool DbUserHasBeenSet() const { return m_dbUserHasBeenSet; }
  template <typename DbUserT = Aws::String>
  void SetDbUser(DbUserT&& value) {
    m_dbUserHasBeenSet = true;
    m_dbUser = std::forward<DbUserT>(value);
  }
  template <typename DbUserT = Aws::String>
  ExecuteStatementRequest& WithDbUser(DbUserT&& value) {
    SetDbUser(std::forward<DbUserT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the database. This parameter is required when authenticating
   * using either Secrets Manager or temporary credentials. </p>
   */
  inline const Aws::String& GetDatabase() const { return m_database; }
  inline bool DatabaseHasBeenSet() const { return m_databaseHasBeenSet; }
  template <typename DatabaseT = Aws::String>
  void SetDatabase(DatabaseT&& value) {
    m_databaseHasBeenSet = true;
    m_database = std::forward<DatabaseT>(value);
  }
  template <typename DatabaseT = Aws::String>
  ExecuteStatementRequest& WithDatabase(DatabaseT&& value) {
    SetDatabase(std::forward<DatabaseT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value that indicates whether to send an event to the Amazon EventBridge
   * event bus after the SQL statement runs. </p>
   */
  inline bool GetWithEvent() const { return m_withEvent; }
  inline bool WithEventHasBeenSet() const { return m_withEventHasBeenSet; }
  inline void SetWithEvent(bool value) {
    m_withEventHasBeenSet = true;
    m_withEvent = value;
  }
  inline ExecuteStatementRequest& WithWithEvent(bool value) {
    SetWithEvent(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the SQL statement. You can name the SQL statement when you create
   * it to identify the query. </p>
   */
  inline const Aws::String& GetStatementName() const { return m_statementName; }
  inline bool StatementNameHasBeenSet() const { return m_statementNameHasBeenSet; }
  template <typename StatementNameT = Aws::String>
  void SetStatementName(StatementNameT&& value) {
    m_statementNameHasBeenSet = true;
    m_statementName = std::forward<StatementNameT>(value);
  }
  template <typename StatementNameT = Aws::String>
  ExecuteStatementRequest& WithStatementName(StatementNameT&& value) {
    SetStatementName(std::forward<StatementNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The parameters for the SQL statement.</p>
   */
  inline const Aws::Vector<SqlParameter>& GetParameters() const { return m_parameters; }
  inline bool ParametersHasBeenSet() const { return m_parametersHasBeenSet; }
  template <typename ParametersT = Aws::Vector<SqlParameter>>
  void SetParameters(ParametersT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters = std::forward<ParametersT>(value);
  }
  template <typename ParametersT = Aws::Vector<SqlParameter>>
  ExecuteStatementRequest& WithParameters(ParametersT&& value) {
    SetParameters(std::forward<ParametersT>(value));
    return *this;
  }
  template <typename ParametersT = SqlParameter>
  ExecuteStatementRequest& AddParameters(ParametersT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters.emplace_back(std::forward<ParametersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The serverless workgroup name or Amazon Resource Name (ARN). This parameter
   * is required when connecting to a serverless workgroup and authenticating using
   * either Secrets Manager or temporary credentials.</p>
   */
  inline const Aws::String& GetWorkgroupName() const { return m_workgroupName; }
  inline bool WorkgroupNameHasBeenSet() const { return m_workgroupNameHasBeenSet; }
  template <typename WorkgroupNameT = Aws::String>
  void SetWorkgroupName(WorkgroupNameT&& value) {
    m_workgroupNameHasBeenSet = true;
    m_workgroupName = std::forward<WorkgroupNameT>(value);
  }
  template <typename WorkgroupNameT = Aws::String>
  ExecuteStatementRequest& WithWorkgroupName(WorkgroupNameT&& value) {
    SetWorkgroupName(std::forward<WorkgroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  ExecuteStatementRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The data format of the result of the SQL statement. If no format is
   * specified, the default is JSON.</p>
   */
  inline ResultFormatString GetResultFormat() const { return m_resultFormat; }
  inline bool ResultFormatHasBeenSet() const { return m_resultFormatHasBeenSet; }
  inline void SetResultFormat(ResultFormatString value) {
    m_resultFormatHasBeenSet = true;
    m_resultFormat = value;
  }
  inline ExecuteStatementRequest& WithResultFormat(ResultFormatString value) {
    SetResultFormat(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of seconds to keep the session alive after the query finishes. The
   * maximum time a session can keep alive is 24 hours. After 24 hours, the session
   * is forced closed and the query is terminated.</p>
   */
  inline int GetSessionKeepAliveSeconds() const { return m_sessionKeepAliveSeconds; }
  inline bool SessionKeepAliveSecondsHasBeenSet() const { return m_sessionKeepAliveSecondsHasBeenSet; }
  inline void SetSessionKeepAliveSeconds(int value) {
    m_sessionKeepAliveSecondsHasBeenSet = true;
    m_sessionKeepAliveSeconds = value;
  }
  inline ExecuteStatementRequest& WithSessionKeepAliveSeconds(int value) {
    SetSessionKeepAliveSeconds(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The session identifier of the query.</p>
   */
  inline const Aws::String& GetSessionId() const { return m_sessionId; }
  inline bool SessionIdHasBeenSet() const { return m_sessionIdHasBeenSet; }
  template <typename SessionIdT = Aws::String>
  void SetSessionId(SessionIdT&& value) {
    m_sessionIdHasBeenSet = true;
    m_sessionId = std::forward<SessionIdT>(value);
  }
  template <typename SessionIdT = Aws::String>
  ExecuteStatementRequest& WithSessionId(SessionIdT&& value) {
    SetSessionId(std::forward<SessionIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_sql;
  bool m_sqlHasBeenSet = false;

  Aws::String m_clusterIdentifier;
  bool m_clusterIdentifierHasBeenSet = false;

  Aws::String m_secretArn;
  bool m_secretArnHasBeenSet = false;

  Aws::String m_dbUser;
  bool m_dbUserHasBeenSet = false;

  Aws::String m_database;
  bool m_databaseHasBeenSet = false;

  bool m_withEvent{false};
  bool m_withEventHasBeenSet = false;

  Aws::String m_statementName;
  bool m_statementNameHasBeenSet = false;

  Aws::Vector<SqlParameter> m_parameters;
  bool m_parametersHasBeenSet = false;

  Aws::String m_workgroupName;
  bool m_workgroupNameHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  ResultFormatString m_resultFormat{ResultFormatString::NOT_SET};
  bool m_resultFormatHasBeenSet = false;

  int m_sessionKeepAliveSeconds{0};
  bool m_sessionKeepAliveSecondsHasBeenSet = false;

  Aws::String m_sessionId;
  bool m_sessionIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace RedshiftDataAPIService
}  // namespace Aws
